//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.2
import QtGraphicalEffects 1.0

import wesual.Controls 1.0
import wesual.data.Core 1.0

FocusScope {
    id : dropDown

    property list<Document>         model
    // TODO: add filter group if needed for multiple search properties
    property StringFilter           filter
    property string                 displayPropertyName : "documentId"
    property bool                   immediateFocus      : false
    property bool                   hoverNeeded         : false // show control borders only on hover

    property alias placeholderText      : textField.placeholderText
    property alias titlebarText         : dropDownPopup.titlebarText
    property alias titlebarClickable    : dropDownPopup.titlebarClickable
    property alias delegate             : dropDownPopup.delegate
    property alias popupWidth           : dropDownPopup.width
    property alias popupHeight          : dropDownPopup.height
    property alias popupWillScroll      : dropDownPopup.willScroll

    readonly property bool open         : dropDownPopup.popupVisible

    property string     currentText     : ""
    property int        currentIndex    : -1

    signal documentChosen(Document chosenDocument)
    signal titlebarClicked()

    function openPopup()  {
        dropDownPopup.activeIndex = currentIndex;
        dropDownPopup.show();
    }

    // Key handling
    focus : true
    Keys.onSpacePressed : {
        if (!dropDownPopup.popupVisible) {
            dropDown.openPopup();
        }
    }
    Keys.onUpPressed : {
        if(textField.activeFocus && dropDownPopup.popupVisible){
            dropDownPopup.forceActiveFocus();
        }
        dropDownPopup.indexDecrement();
    }
    Keys.onDownPressed : {
        if (!dropDownPopup.popupVisible) {
            dropDown.openPopup();
        }

        if(textField.activeFocus){
            dropDownPopup.forceActiveFocus();
        }
        dropDownPopup.indexIncrement();
    }
    Keys.onReturnPressed : dropDownPopup.acceptValidIndex();
    Keys.onEnterPressed  : dropDownPopup.acceptValidIndex();
    Keys.onRightPressed : {}
    Keys.onLeftPressed : {}
    Keys.onPressed : {
        if (event.key === Qt.Key_PageUp || event.key === Qt.Key_PageDown) {
            // do not propagate navigation keys while focus item
            event.accepted = true;
        }
    }

    width : 100
    implicitHeight   : 26

    activeFocusOnTab : true

    Rectangle {
        id : topcontainer

        anchors.fill : parent
        border {
            width : hoverNeeded &&
                    !mouseArea.containsMouse &&
                    !textField.activeFocus ? 0 : 1
            color : {
                if (mouseArea.pressed || textField.activeFocus) {
                    return UI.color(UI.ControlBorderPress);
                } else if (mouseArea.containsMouse) {
                    return UI.color(UI.ControlBorderHover);
                }
                return UI.color(UI.PrimaryControlBorder);
            }
        }

        Behavior on border.color {
            ColorAnimation { duration : 0 }
        }

        TextField {
            id : textField

            Component.onCompleted : {
                if  (immediateFocus)
                    forceActiveFocus();
            }

            Timer {
                id : timer;
                interval : 500;
                running : false;
                repeat : false;

                // when input delay elapsed start filtering
                onTriggered : {
                    if (textField.activeFocus){
                        filter.propertyFilter = textField.text;
                        dropDown.currentIndex = -1;
                    }

                    if (dropDownPopup.model.length && !dropDown.open)
                        openPopup();

                    if (dropDownPopup.model.length === 0) {
                        dropDownPopup.hide();
                        topcontainer.border.color =
                                UI.color(UI.PrimaryControlBorder);
                    } else
                        topcontainer.border.color =
                                UI.color(UI.SecondaryControlBorder);
                }
            }

            backgroundColor : topcontainer.color
            border.width : 0
            anchors {
                fill : parent
                margins : topcontainer.border.width
                leftMargin : 2
                right : indicator.left
                rightMargin: indicator.width + 8
            }
            placeholderText : {
                if(!textField.activeFocus && dropDown.currentText){
                    return dropDown.currentText;
                } else {
                    return "";
                }
            }
            textColor : text.length ?
                        UI.color(UI.PrimaryTextColor) :
                        UI.color(UI.SecondaryTextColor)

            // search input
            onTextChanged : {
                if(textField.activeFocus)
                    timer.restart();

                if (activeFocus && !dropDownPopup.popupVisible)
                    dropDown.openPopup();
            }
        }

        ColorizedImage {
            id : indicator
            anchors {
                right : parent.right
                rightMargin : 6
                verticalCenter : parent.verticalCenter
                verticalCenterOffset : 1
            }
            source : "qrc:/controls/images/dropdown-indicator.png"
            color : (mouseArea.indicatorHovered || dropDownPopup.popupVisible) ?
                                                            "#54A57E" : "#CCCCC"

            opacity : {
                if (hoverNeeded) {
                    if (dropDownPopup.popupVisible)
                        return 1.0;
                    else if (mouseArea.containsMouse || textField.activeFocus)
                        return 0.2;
                    else
                        return 0.0;
                } else {
                    if (dropDownPopup.popupVisible)
                        return 1.0;
                    else
                        return 0.2;
                }
            }
        }
    }

    function reset() {
        dropDown.focus  = true;
        textField.text  = "";
        filter.propertyFilter = "";
        dropDownPopup.hide();
    }

    MouseArea {
        id : mouseArea

        readonly property bool indicatorHovered :
            (mouseX > textField.width) && mouseArea.containsMouse

        hoverEnabled : true
        anchors.fill : parent
        acceptedButtons : Qt.LeftButton
        onPressed : {
            if (!indicatorHovered)
                textField.forceActiveFocus();
            else
                openPopup();
        }
    }

    SuggestionBoxPopup {
        id : dropDownPopup
        model : DocumentModel {
            documents : dropDown.model
        }
        inputBox : dropDown
        displayPropertyName : dropDown.displayPropertyName

        onPopupVisibleChanged : {
            if (!popupVisible) {
                reset();
            }
        }

        onAccepted : {
            dropDown.currentIndex = index
            documentChosen(dropDown.model[index])
            currentText = dropDown.model[index][displayPropertyName];
            reset();
        }
        onTitlebarClicked : dropDown.titlebarClicked();
    }
}
